<?php

namespace App\Http\Controllers\Reporter;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\NewsPost;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;

class NewsController extends Controller
{
    public function index()
    {
        $news = NewsPost::where('user_id', Auth::id())->latest()->get();
        return view('reporter.news.index', compact('news'));
    }

    public function create()
    {
        $categories = Category::with('subcategories')->get();
        return view('reporter.news.create', compact('categories'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required',
            'category_id' => 'required',
            'subcategory_id' => 'nullable|exists:subcategories,id',
            'content' => 'required',
        ]);

        $imagePath = null;
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('news', 'public');
        }

        NewsPost::create([
            'user_id' => Auth::id(),
            'category_id' => $request->category_id,
            'subcategory_id' => $request->subcategory_id,
            'title' => $request->title,
            'slug' => Str::slug($request->title),
            'content' => $request->content,
            'image' => $imagePath,
            'status' => 'pending',
            'meta_title' => $request->meta_title ?? $request->title,
        ]);

        return redirect()->route('reporter.news.index')->with('success', 'News submitted for approval');
    }

    public function idCard()
    {
        $user = Auth::user();
        if (!$user->profile_photo) {
            return back()->with('error', 'Please upload a profile photo first.');
        }
        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('reporter.docs.idcard', compact('user'));
        return $pdf->download('id-card-'.$user->id.'.pdf');
    }

    public function appointmentLetter()
    {
        $user = Auth::user();
        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('reporter.docs.appointment', compact('user'));
        return $pdf->download('appointment-letter-'.$user->id.'.pdf');
    }

    public function edit(string $id)
    {
        $news = NewsPost::where('id', $id)->where('user_id', Auth::id())->firstOrFail();
        $categories = Category::with('subcategories')->get();
        return view('reporter.news.edit', compact('news', 'categories'));
    }

    public function update(Request $request, string $id)
    {
        $news = NewsPost::where('id', $id)->where('user_id', Auth::id())->firstOrFail();
        
        $request->validate([
            'title' => 'required',
            'category_id' => 'required',
            'content' => 'required',
        ]);

        $imagePath = $news->image;
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('news', 'public');
        }

        $news->update([
            'category_id' => $request->category_id,
            'subcategory_id' => $request->subcategory_id,
            'title' => $request->title,
            'slug' => Str::slug($request->title),
            'content' => $request->content,
            'image' => $imagePath,
            'status' => 'pending', // Re-approval required
            'meta_title' => $request->meta_title ?? $request->title,
        ]);

        return redirect()->route('reporter.news.index')->with('success', 'News post updated and sent for review.');
    }

    public function destroy(string $id)
    {
        $news = NewsPost::where('id', $id)->where('user_id', Auth::id())->firstOrFail();
        $news->delete();
        return redirect()->route('reporter.news.index')->with('success', 'News post deleted.');
    }
}
