<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;

class ReporterController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $reporters = User::role('reporter')->latest()->get();
        return view('admin.reporters.index', compact('reporters'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.reporters.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
        ]);

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => bcrypt($request->password),
            'status' => 'pending', // Admins approve reporters
        ]);

        $user->assignRole('reporter');

        return redirect()->route('admin.reporters.index')->with('success', 'Reporter created but pending approval.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $reporter = User::findOrFail($id);
        return view('admin.reporters.show', compact('reporter'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $reporter = User::findOrFail($id);
        return view('admin.reporters.edit', compact('reporter'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $reporter = User::findOrFail($id);
        $request->validate([
            'status' => 'required|in:active,pending,suspended',
        ]);

        $oldStatus = $reporter->status;
        $reporter->update([
            'status' => $request->status,
        ]);

        if ($oldStatus != 'active' && $request->status == 'active') {
            $reporter->notify(new \App\Notifications\AccountActivatedNotification($reporter));
        }

        return redirect()->route('admin.reporters.index')->with('success', 'Reporter status updated.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $reporter = User::findOrFail($id);
        $reporter->delete();

        return redirect()->route('admin.reporters.index')->with('success', 'Reporter deleted.');
    }
}
